#!/usr/local/bin/python

from Tkinter  import *                                 # Tk widgets
from guitools import frame, label, button, entry       # widget builders
from guimixin import GuiMixin                          # common methods

class FormGui(GuiMixin, Frame):
    def __init__(self, mapping):                       # an extended frame
        Frame.__init__(self)                           # on default top-level
        self.pack(expand=YES, fill=BOTH)               # all parts expandable
        self.master.title('Python Browser 0.1')       
        self.master.iconname("pbrowse")
        self.makeMainBox()
        self.table     = mapping               # a dict, dbm, shelve,..
        self.index     = mapping.keys()        # list of table keys
        self.cursor    = -1                    # current index position
        self.currslots = []                    # current form's (key,text)'s
        self.currform  = None                  # current form window
        self.listbox   = None                  # index listbox window

    def makeMainBox(self):
        frm = frame(self, TOP)
        frm.config(bd=2)
        button(frm, LEFT, 'next',  self.onNext)       # next in list
        button(frm, LEFT, 'prev',  self.onPrev)       # backup in list
        button(frm, LEFT, 'find',  self.onFind)       # find from key
        frm = frame(self, TOP)
        self.keytext = StringVar()                    # current record's key
        label(frm, LEFT, 'KEY=>')                     # change before 'find'
        entry(frm, LEFT,  self.keytext)             
        frm = frame(self, TOP)
        frm.config(bd=2)
        button(frm,  LEFT,  'store',  self.onStore)     # updated entry data
        button(frm,  LEFT,  'new',    self.onNew)       # clear fields
        button(frm,  LEFT,  'index',  self.onMakeList)  # show key list
        button(frm,  LEFT,  'delete', self.onDelete)    # show key list
        button(self, BOTTOM,'quit',   self.quit)        # from guimixin

    def onPrev(self):
        if self.cursor <= 0:
            self.infobox('Backup', "Front of table")
        else:
            self.cursor = self.cursor - 1
            self.display()

    def onNext(self):
        if self.cursor >= len(self.index)-1:
            self.infobox('Advance', "End of table")
        else:
            self.cursor = self.cursor + 1
            self.display()

    def sameKeys(self, item):
        keys1 = item.keys()
        keys2 = map(lambda x:x[0], self.currslots)
        keys1.sort(); keys2.sort()              # keys list order differs
        return keys1 == keys2                   # if insertion-order differs

    def display(self):
        key = self.index[self.cursor]
        self.keytext.set(key)                   # change key in main box 
        item = self.table[key]                  # in dict, dbm, shelf, classes
        if self.sameKeys(item): 
            self.currform.title('Key: ' + `key`)  
            for (field, text) in self.currslots:
                text.set(`item[field]`)         # same fields? reuse form
        else:
            if self.currform:
                self.currform.destroy()         # different fields?  
            new = Toplevel()                    # replace current box
            new.title('Key: ' + `key`)          # new resizable window
            new.iconname("pform")
            left  = frame(new, LEFT)
            right = frame(new, RIGHT)
            self.currslots = []                 # list of (field, entry)
            for field in item.keys():
                label(left, TOP, `field`)       # key,value to strings
                text = StringVar()              # we could sort keys here
                text.set( `item[field]` )
                entry(right, TOP, text)
                self.currslots.append((field, text))
            self.currform = new
            new.protocol('WM_DELETE_WINDOW', lambda:0)   # ignore destroy's
        self.selectlist()                                # update listbox

    def onStore(self):
        if not self.currform: return
        key = self.keytext.get()
        if key in self.index:                   # change existing record
            item = self.table[key]              # not: self.table[key][field]=
        else:
            item = {}                           # create a new record
            self.index.append(key)              # add to index and listbox
            if self.listbox: 
                self.listbox.insert(len(self.index)-1, key) 
        for (field, text) in self.currslots:
            try:
                item[field] = eval(text.get())  # convert back from string
            except:
                self.errorbox('Bad data: "%s" = "%s"' % (field, text.get()))
                item[field] = None
        self.table[key] = item                  # add to dict, dbm, shelf,...
        self.onFind(key)                        # readback: set cursor,listbox

    def onNew(self):
        if not self.currform: return
        self.keytext.set('?%d' % len(self.index))  # default key unless typed
        for (field, text) in self.currslots:       # clear key/fields for entry
            text.set('') 
        self.currform.title('Key: ?')  
 
    def onFind(self, key=None):
        target = key or self.keytext.get()            # passed in, or entered
        try:
            self.cursor = self.index.index(target)    # in keys list?
            self.display()
        except:
            self.infobox('Not found', "Key doesn't exist", 'info')

    def onDelete(self):
        if not self.currform or not self.index: return
        currkey = self.index[self.cursor]
        del self.table[currkey]                      # table, index, listbox
        del self.index[self.cursor:self.cursor+1]    # like "list[i:i+1] = []"
        if self.listbox: 
            self.listbox.delete(self.cursor)         # delete from listbox 
        if self.cursor < len(self.index):
            self.display()                           # show next record if any 
        elif self.cursor > 0:
            self.cursor = self.cursor-1              # show prior if delete end
            self.display()
        else:                                        # leave box if delete last
            self.onNew() 

    def onList(self,evnt):
        if not self.index: return                  # empty
        index = self.listbox.curselection()        # on listbox double-click
        label = self.listbox.get(index)            # fetch selected key text
        self.onFind(label)                         # and call method here

    def onMakeList(self):
        if self.listbox: return             # already up?
        new = Toplevel()                    # new resizable window
        new.title("Table Key Index")        # select keys from a listbox
        new.iconname("pindex")
        frm    = frame(new, TOP)
        scroll = Scrollbar(frm)
        list   = Listbox(frm)
        list.pack(side=LEFT, expand=YES, fill=BOTH)
        list.config(yscrollcommand=scroll.set, relief=SUNKEN)
        scroll.pack(side=RIGHT, fill=BOTH)
        scroll.config(command=list.yview, relief=SUNKEN)
        pos = 0
        for key in self.index:                       # add to list-box
            list.insert(pos, key)                    # sort list fist?
            pos = pos + 1
        list.config(selectmode=SINGLE, setgrid=1)    # select,resize modes
        list.bind('<Double-1>', self.onList)         # on double-clicks
        self.listbox = list
        if pos > 0 and self.cursor >= 0:             # highlight position
            self.selectlist()
        new.protocol('WM_DELETE_WINDOW', lambda:0)   # ignore destroy's

    def selectlist(self):
        if self.listbox:         
            self.listbox.select_clear(0, self.listbox.size())
            self.listbox.select_set(self.cursor)     # list box tracks cursor


if __name__ == '__main__': 
    from formtest import cast                   # self-test code
    for k in cast.keys(): print k, cast[k]      # get dict of dict's
    FormGui(cast).mainloop()
    for k in cast.keys(): print k, cast[k]      # show modified table on exit
