package org.apereo.cas.adaptors.yubikey.web.flow;

import module java.base;
import org.apereo.cas.adaptors.yubikey.YubiKeyCredential;
import org.apereo.cas.configuration.CasConfigurationProperties;
import org.apereo.cas.util.CollectionUtils;
import org.apereo.cas.web.flow.CasWebflowConstants;
import org.apereo.cas.web.flow.configurer.AbstractCasMultifactorWebflowConfigurer;
import org.apereo.cas.web.flow.configurer.CasMultifactorWebflowCustomizer;
import lombok.val;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.webflow.definition.registry.FlowDefinitionRegistry;
import org.springframework.webflow.engine.builder.support.FlowBuilderServices;

/**
 * This is {@link YubiKeyMultifactorWebflowConfigurer}.
 *
 * @author Misagh Moayyed
 * @since 5.0.0
 */
public class YubiKeyMultifactorWebflowConfigurer extends AbstractCasMultifactorWebflowConfigurer {

    /**
     * Webflow event id.
     */
    public static final String MFA_YUBIKEY_EVENT_ID = "mfa-yubikey";

    public YubiKeyMultifactorWebflowConfigurer(final FlowBuilderServices flowBuilderServices,
                                               final FlowDefinitionRegistry flowDefinitionRegistry,
                                               final FlowDefinitionRegistry mfaFlowDefinitionRegistry,
                                               final ConfigurableApplicationContext applicationContext,
                                               final CasConfigurationProperties casProperties,
                                               final List<CasMultifactorWebflowCustomizer> mfaFlowCustomizers) {
        super(flowBuilderServices, flowDefinitionRegistry, applicationContext,
            casProperties, Optional.of(mfaFlowDefinitionRegistry), mfaFlowCustomizers);
    }

    @Override
    protected void doInitialize() {
        val yubiProps = casProperties.getAuthn().getMfa().getYubikey();
        multifactorAuthenticationFlowDefinitionRegistries.stream()
            .map(registry -> getFlow(registry, MFA_YUBIKEY_EVENT_ID))
            .forEach(flow -> {
                createFlowVariable(flow, CasWebflowConstants.VAR_ID_CREDENTIAL, YubiKeyCredential.class);
                flow.getStartActionList().add(createEvaluateAction(CasWebflowConstants.ACTION_ID_INITIAL_FLOW_SETUP));
                createEndState(flow, CasWebflowConstants.STATE_ID_SUCCESS);
                val initLoginFormState = createActionState(flow, CasWebflowConstants.STATE_ID_INIT_LOGIN_FORM,
                    createEvaluateAction(CasWebflowConstants.ACTION_ID_YUBIKEY_PREPARE_LOGIN),
                    createEvaluateAction(CasWebflowConstants.ACTION_ID_INIT_LOGIN_ACTION));
                createTransitionForState(initLoginFormState, CasWebflowConstants.TRANSITION_ID_SUCCESS, CasWebflowConstants.STATE_ID_CHECK_ACCOUNT_REGISTRATION);
                setStartState(flow, initLoginFormState);
                val acctRegCheckState = createActionState(flow, CasWebflowConstants.STATE_ID_CHECK_ACCOUNT_REGISTRATION,
                    createEvaluateAction(CasWebflowConstants.ACTION_ID_YUBIKEY_ACCOUNT_REGISTRATION));
                createTransitionForState(acctRegCheckState, CasWebflowConstants.TRANSITION_ID_REGISTER, CasWebflowConstants.STATE_ID_VIEW_REGISTRATION);
                createTransitionForState(acctRegCheckState, CasWebflowConstants.TRANSITION_ID_SUCCESS, CasWebflowConstants.STATE_ID_VIEW_LOGIN_FORM);
                val saveState = createActionState(flow, CasWebflowConstants.STATE_ID_YUBIKEY_SAVE_REGISTRATION,
                    createEvaluateAction(CasWebflowConstants.ACTION_ID_YUBIKEY_SAVE_ACCOUNT_REGISTRATION));
                createTransitionForState(saveState, CasWebflowConstants.TRANSITION_ID_SUCCESS, CasWebflowConstants.STATE_ID_VIEW_LOGIN_FORM);
                createTransitionForState(saveState, CasWebflowConstants.TRANSITION_ID_ERROR, CasWebflowConstants.STATE_ID_INIT_LOGIN_FORM);
                val realSubmitState = createActionState(flow, CasWebflowConstants.STATE_ID_REAL_SUBMIT,
                    createEvaluateAction(CasWebflowConstants.ACTION_ID_YUBIKEY_AUTHENTICATION));
                createTransitionForState(realSubmitState, CasWebflowConstants.TRANSITION_ID_SUCCESS, CasWebflowConstants.STATE_ID_SUCCESS);
                createTransitionForState(realSubmitState, CasWebflowConstants.TRANSITION_ID_ERROR, CasWebflowConstants.STATE_ID_VIEW_LOGIN_FORM);
                val setPrincipalAction = createSetAction("viewScope.principal", "conversationScope.authentication.principal");
                val viewRegState = createViewState(flow, CasWebflowConstants.STATE_ID_VIEW_REGISTRATION, "yubikey/casYubiKeyRegistrationView");
                viewRegState.getEntryActionList().addAll(setPrincipalAction);
                createTransitionForState(viewRegState, CasWebflowConstants.TRANSITION_ID_SUBMIT, CasWebflowConstants.STATE_ID_YUBIKEY_SAVE_REGISTRATION);
                val loginProperties = CollectionUtils.wrapList("token");
                val loginBinder = createStateBinderConfiguration(loginProperties);
                val viewLoginFormState = createViewState(flow, CasWebflowConstants.STATE_ID_VIEW_LOGIN_FORM, "yubikey/casYubiKeyLoginView", loginBinder);
                createStateModelBinding(viewLoginFormState, CasWebflowConstants.VAR_ID_CREDENTIAL, YubiKeyCredential.class);
                viewLoginFormState.getEntryActionList().addAll(setPrincipalAction);
                if (yubiProps.isMultipleDeviceRegistrationEnabled()) {
                    createTransitionForState(viewLoginFormState, CasWebflowConstants.TRANSITION_ID_REGISTER, CasWebflowConstants.STATE_ID_VIEW_REGISTRATION,
                        createTransitionAttributes(false, false));
                }
                createTransitionForState(viewLoginFormState, CasWebflowConstants.TRANSITION_ID_SUBMIT,
                    CasWebflowConstants.STATE_ID_REAL_SUBMIT, createTransitionAttributes(true, true));
            });

        registerMultifactorProviderAuthenticationWebflow(getLoginFlow(), MFA_YUBIKEY_EVENT_ID, yubiProps.getId());
    }
}
