package org.apereo.cas.authentication;

import module java.base;
import org.apereo.cas.authentication.principal.Service;
import org.apereo.cas.authentication.principal.ServiceFactory;
import org.apereo.cas.authentication.principal.WebApplicationService;
import org.apereo.cas.services.ServicesManager;
import org.apereo.cas.util.function.FunctionUtils;
import org.apereo.cas.ws.idp.WSFederationConstants;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.hc.core5.http.NameValuePair;
import org.apache.hc.core5.net.URIBuilder;
import org.jooq.lambda.Unchecked;
import org.jspecify.annotations.Nullable;

/**
 * This is {@link WSFederationAuthenticationServiceSelectionStrategy}.
 *
 * @author Misagh Moayyed
 * @since 5.1.0
 */
@Slf4j
@Getter
@Setter
public class WSFederationAuthenticationServiceSelectionStrategy extends BaseAuthenticationServiceSelectionStrategy {
    @Serial
    private static final long serialVersionUID = 8035218407906419228L;

    public WSFederationAuthenticationServiceSelectionStrategy(final ServicesManager servicesManager,
                                                              final ServiceFactory<WebApplicationService> webApplicationServiceFactory) {
        super(servicesManager, webApplicationServiceFactory);
    }

    private static Optional<NameValuePair> getRealmAsParameter(final Service service) {
        return FunctionUtils.doUnchecked(() -> {
            val builder = new URIBuilder(service.getId());
            return builder.getQueryParams()
                .stream()
                .filter(p -> p.getName().equals(WSFederationConstants.WTREALM))
                .findFirst();
        });
    }

    private static Optional<NameValuePair> getReplyAsParameter(@Nullable final Service service) {
        return Optional.ofNullable(service)
            .map(Unchecked.function(svc -> {
                val builder = new URIBuilder(Objects.requireNonNull(svc).getId());
                return builder.getQueryParams()
                    .stream()
                    .filter(p -> p.getName().equals(WSFederationConstants.WREPLY))
                    .findFirst()
                    .orElse(null);
            }));
    }

    @Override
    public @Nullable Service resolveServiceFrom(@Nullable final Service service) {
        val replyParamRes = getReplyAsParameter(service);
        if (replyParamRes.isPresent()) {
            val serviceReply = replyParamRes.get().getValue();
            LOGGER.trace("Located service id [{}] from service authentication request at [{}]",
                serviceReply, Objects.requireNonNull(service).getId());
            return createService(serviceReply, service);
        }
        LOGGER.trace("Resolved final service as [{}]", service);
        return service;
    }

    @Override
    public boolean supports(@Nullable final Service service) {
        if (service == null) {
            LOGGER.trace("Provided service is undefined");
            return false;
        }
        LOGGER.trace("Evaluating service requested identified as [{}]", service.getId());
        val realmAsParameter = getRealmAsParameter(service);
        if (realmAsParameter.isEmpty()) {
            LOGGER.trace("Parameter [{}] is undefined in the request", WSFederationConstants.WTREALM);
            return false;
        }
        return getReplyAsParameter(service).isPresent();
    }
}
