package org.apereo.cas.webauthn.storage;

import module java.base;
import org.apereo.cas.configuration.CasConfigurationProperties;
import org.apereo.cas.util.crypto.CipherExecutor;
import org.apereo.cas.util.function.FunctionUtils;
import org.apereo.cas.webauthn.WebAuthnUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.yubico.data.CredentialRegistration;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.io.Resource;

/**
 * This is {@link JsonResourceWebAuthnCredentialRepository}.
 *
 * @author Misagh Moayyed
 * @since 6.3.0
 */
@Slf4j
public class JsonResourceWebAuthnCredentialRepository extends BaseWebAuthnCredentialRepository implements InitializingBean {
    private final Resource location;

    public JsonResourceWebAuthnCredentialRepository(final CasConfigurationProperties properties,
                                                    final Resource location,
                                                    final CipherExecutor<String, String> cipherExecutor) {
        super(properties, cipherExecutor);
        this.location = location;
    }

    @Override
    public void afterPropertiesSet() {
        readFromJsonRepository();
    }

    @Override
    public Collection<CredentialRegistration> getRegistrationsByUsername(final String username) {
        val storage = readFromJsonRepository();
        return storage.containsKey(username.trim().toLowerCase(Locale.ENGLISH))
            ? storage.get(username.trim().toLowerCase(Locale.ENGLISH))
            : new HashSet<>();
    }

    @Override
    public Stream<CredentialRegistration> stream() {
        return readFromJsonRepository().values().stream().flatMap(Collection::stream);
    }

    @Override
    protected void update(final String username, final Collection<CredentialRegistration> givenRecords) {
        val storage = readFromJsonRepository();
        val records = givenRecords.stream()
            .map(record -> {
                if (record.getRegistrationTime() == null) {
                    return record.withRegistrationTime(Instant.now(Clock.systemUTC()));
                }
                return record;
            }).toList();
        storage.put(username.trim().toLowerCase(Locale.ENGLISH), new LinkedHashSet<>(records));
        FunctionUtils.doUnchecked(_ -> WebAuthnUtils.getObjectMapper().writerWithDefaultPrettyPrinter().writeValue(location.getFile(), storage));
    }

    private Map<String, Set<CredentialRegistration>> readFromJsonRepository() {
        return FunctionUtils.doUnchecked(() -> {
            LOGGER.trace("Ensuring JSON repository file exists at [{}]", location.getFile());
            val result = location.getFile().createNewFile();
            if (result) {
                LOGGER.trace("Created JSON repository file at [{}]", location.getFile());
            }
            if (location.getFile().length() > 0) {
                LOGGER.trace("Reading JSON repository file at [{}]", location.getFile());
                return new ConcurrentHashMap<>(WebAuthnUtils.getObjectMapper().readValue(location.getFile(), new TypeReference<>() {
                }));
            }
            return new ConcurrentHashMap<>(0);
        });
    }
}
