package org.apereo.cas.support.saml.services;

import module java.base;
import org.apereo.cas.authentication.BaseAuthenticationServiceSelectionStrategy;
import org.apereo.cas.authentication.principal.Service;
import org.apereo.cas.authentication.principal.ServiceFactory;
import org.apereo.cas.authentication.principal.WebApplicationService;
import org.apereo.cas.services.ServicesManager;
import org.apereo.cas.support.saml.SamlProtocolConstants;
import org.apereo.cas.util.function.FunctionUtils;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.hc.core5.http.NameValuePair;
import org.apache.hc.core5.net.URIBuilder;

/**
 * This is {@link SamlIdPEntityIdAuthenticationServiceSelectionStrategy}.
 *
 * @author Misagh Moayyed
 * @since 5.0.0
 */
@Slf4j
@Setter
@Getter
public class SamlIdPEntityIdAuthenticationServiceSelectionStrategy extends BaseAuthenticationServiceSelectionStrategy {
    @Serial
    private static final long serialVersionUID = -2059445756475980894L;

    private final String casServiceUrlPattern;

    public SamlIdPEntityIdAuthenticationServiceSelectionStrategy(final ServicesManager servicesManager,
        final ServiceFactory<WebApplicationService> webApplicationServiceFactory,
        final String casServerPrefix) {
        super(servicesManager, webApplicationServiceFactory);
        this.casServiceUrlPattern = "^".concat(casServerPrefix).concat(".*");
    }

    @Override
    public Service resolveServiceFrom(final Service service) {
        val entityId = getEntityIdAsParameter(service).orElseThrow().getValue();
        LOGGER.trace("Located entity id [{}] from service authentication request at [{}]", entityId, service.getId());
        return createService(entityId, service);
    }

    @Override
    public boolean supports(final Service service) {
        return service != null && service.getId().matches(this.casServiceUrlPattern)
            && getEntityIdAsParameter(service).isPresent();
    }

    /**
     * Gets entity id as parameter.
     *
     * @param service the service
     * @return the entity id as parameter
     */
    protected static Optional<NameValuePair> getEntityIdAsParameter(final Service service) {
        return FunctionUtils.doAndHandle(() -> {
            val builder = new URIBuilder(service.getId());
            return builder.getQueryParams()
                .stream()
                .filter(p -> p.getName().equals(SamlProtocolConstants.PARAMETER_ENTITY_ID))
                .findFirst();
        });
    }
}
