package org.apereo.cas.services.locator;

import module java.base;
import org.apereo.cas.services.RegisteredService;
import org.apereo.cas.util.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.jgit.util.FileUtils;

/**
 * This is {@link GitRepositoryRegisteredServiceLocator}.
 *
 * @author Misagh Moayyed
 * @since 6.3.0
 */
@FunctionalInterface
public interface GitRepositoryRegisteredServiceLocator {
    /**
     * The File extensions.
     */
    List<String> FILE_EXTENSIONS = CollectionUtils.wrapList("json", "yaml", "yml");

    /**
     * The constant PATTERN_ACCEPTED_REPOSITORY_FILES to optionally be combined with root-directory prefix.
     */
    String PATTEN_ACCEPTED_REPOSITORY_FILES = ".+\\.(" + String.join("|", FILE_EXTENSIONS) + ')';

    /**
     * Determine file.
     *
     * @param service   the service
     * @param extension the extension
     * @return the file
     */
    File determine(RegisteredService service, String extension);

    /**
     * Locate file that is linked to the service.
     *
     * @param service the service
     * @return the optional
     */
    default Optional<File> locate(final RegisteredService service) {
        return FILE_EXTENSIONS.stream()
            .map(ext -> determine(service, ext))
            .filter(File::exists)
            .findFirst();
    }

    /**
     * Normalize parent directory file.
     *
     * @param repositoryDirectory the repository directory
     * @param rootDirectory       the root directory
     * @return the file
     * @throws IOException the io exception
     */
    static File normalizeParentDirectory(final File repositoryDirectory, final String rootDirectory) throws IOException {
        var parentDirectory = repositoryDirectory;
        if (StringUtils.isNotBlank(rootDirectory)) {
            parentDirectory = new File(repositoryDirectory, rootDirectory);
            FileUtils.mkdir(parentDirectory, true);
        }
        return parentDirectory;
    }
}
