package org.apereo.cas;

import module java.base;
import org.apereo.cas.config.CasJdbcCloudConfigBootstrapAutoConfiguration;
import org.apereo.cas.configuration.CasConfigurationProperties;
import org.apereo.cas.configuration.api.MutablePropertySource;
import org.apereo.cas.configuration.model.support.jpa.AbstractJpaProperties;
import org.apereo.cas.configuration.support.JpaBeans;
import org.apereo.cas.test.CasTestExtension;
import org.apereo.cas.util.spring.boot.SpringBootTestAutoConfigurations;
import lombok.val;
import org.apache.commons.io.FileUtils;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.cloud.bootstrap.config.BootstrapPropertySource;
import org.springframework.core.env.ConfigurableEnvironment;
import static org.junit.jupiter.api.Assertions.*;

/**
 * This is {@link CasJdbcCloudConfigBootstrapAutoConfigurationTests}.
 *
 * @author Misagh Moayyed
 * @since 6.0.0
 */
@EnableConfigurationProperties(CasConfigurationProperties.class)
@SpringBootTestAutoConfigurations
@SpringBootTest(classes = CasJdbcCloudConfigBootstrapAutoConfiguration.class)
@Tag("JDBC")
@ExtendWith(CasTestExtension.class)
class CasJdbcCloudConfigBootstrapAutoConfigurationTests {
    private static final String STATIC_AUTHN_USERS = "casuser::WHATEVER";
    private static final String DB_FILE = System.getProperty("java.io.tmpdir") + "/hsqldb/cas-hsqldb-cloud-config";
    private static final String DB_URL = "jdbc:hsqldb:file:" + DB_FILE + ";shutdown=true;hsqldb.lock_file=false";

    @Autowired
    private CasConfigurationProperties casProperties;

    @Autowired
    private ConfigurableEnvironment environment;


    static {
        val path = Path.of(DB_FILE).getParent();
        if (Files.exists(path)) {
            try {
                Files.walk(path)
                    .sorted(Comparator.reverseOrder())
                    .forEach(p -> {
                        FileUtils.deleteQuietly(p.toFile());
                    });
            } catch (final IOException e) {
                throw new RuntimeException(e);
            }
        }
        System.setProperty(CasJdbcCloudConfigBootstrapAutoConfiguration.CAS_CONFIGURATION_PREFIX + ".url", DB_URL);
    }

    @BeforeAll
    public static void initialize() throws Exception {
        val jpa = new Jpa().setUrl(DB_URL);
        val ds = JpaBeans.newDataSource(jpa);
        try (val connection = ds.getConnection();
             val statement = connection.createStatement()) {
            connection.setAutoCommit(true);
            statement.execute(
                """
                    create table CAS_SETTINGS_TABLE (
                        id BIGINT GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
                        name VARCHAR(255) NOT NULL,
                        value VARCHAR(255),
                        CONSTRAINT uk_cas_settings_name UNIQUE (name)
                    );
                    """);
            statement.execute("insert into CAS_SETTINGS_TABLE (name, value) values('cas.authn.accept.users', '" + STATIC_AUTHN_USERS + "');");
        }
    }

    @Test
    void verifyOperation() {
        assertEquals(STATIC_AUTHN_USERS, casProperties.getAuthn().getAccept().getUsers());

        val propertySource = environment.getPropertySources()
            .stream()
            .filter(source -> source instanceof BootstrapPropertySource<?>)
            .map(BootstrapPropertySource.class::cast)
            .map(BootstrapPropertySource::getDelegate)
            .filter(MutablePropertySource.class::isInstance)
            .map(MutablePropertySource.class::cast)
            .findFirst()
            .orElseThrow();
        propertySource.setProperty("cas.server.prefix", "https://example.org/cas");
        propertySource.setProperty("cas.server.prefix", "https://sso.example.org/cas");
        assertEquals("https://sso.example.org/cas", propertySource.getProperty("cas.server.prefix"));
        propertySource.removeProperty("cas.server.prefix");
        assertNull(propertySource.getProperty("cas.server.prefix"));
        propertySource.removeAll();
        assertEquals(0, propertySource.getPropertyNames().length);
    }

    static class Jpa extends AbstractJpaProperties {
        @Serial
        private static final long serialVersionUID = 1210163210567513705L;
    }
}
