---
layout: default
title: CAS - QR Code Authentication
category: Authentication
---
{% include variables.html %}


# QR Code Authentication

QR Code authentication is a strategy that allows the user to scan a QR code, generated by the
CAS server, using a mobile device and subsequently login after having successfully validated it.

The QR code contains a special identifier embedded within that allows the mobile device to establish a communication
channel using web sockets to the CAS server. Once established, the mobile device may collect credentials from the user
and submit those to CAS for verification. The return result, expected as a `JWT` that is generated by CAS,
is then passed along to the web socket channel for verification and successful login. Subsequent login attempts can allow
for a completely *passwordless* scenario such that the mobile device can continue to reuse the JWT
for authentication attempts, allowing the end user to *scan and proceed*.

Support is enabled by including the following dependency in the WAR overlay:

{% include_cached casmodule.html group="org.apereo.cas" module="cas-server-support-qr-authentication" %}


## Actuator Endpoints

The following endpoints are provided by CAS:

{% include_cached actuators.html endpoints="qrDevices"  %}

## Configuration

{% include_cached casproperties.html properties="cas.authn.qr" %}

## Web Socket Communication

The process of connecting to a web socket connection certainly varies for each mobile app framework. At a high level, 
mobile devices should establish a web socket connection to the CAS server via the `/cas/qr-websocket` endpoint.
The payload must then be sent to the `/qr/accept` path as a map and must contain a `token` field that 
carries the pre-authenticated JWT. The payload header must point to the channel id obtained 
from the QR code under the header name `QR_AUTHENTICATION_CHANNEL_ID` as well 
as the authorized device identifier under `QR_AUTHENTICATION_DEVICE_ID`.

The following code snippet demonstrates this process as an example:

```javascript 
let socket = new SockJS('https://sso.example.org/cas/qr-websocket');
let stompClient = Stomp.over(socket);
let payload = JSON.stringify({'token': '...'});   
let channelId = "...";      
let deviceId = "...";
stompClient.send("/qr/accept", 
    {'QR_AUTHENTICATION_CHANNEL_ID': channelId, 
     'QR_AUTHENTICATION_DEVICE_ID': deviceId}, 
    payload);
```   

The following code snippet demonstrates this process for 
the Android platform based on [StompProtocolAndroid](https://github.com/NaikSoftware/StompProtocolAndroid):

```java
import ua.naiksoftware.stomp.*;
import ua.naiksoftware.stomp.dto.*;

var jwt = "...";
var jsonWebToken = new JSONStringer().object()
  .key("token").value(jwt).endObject();

var channel = "...";      
var deviceId = "...";
var headers = new ArrayList<>();
headers.add(new StompHeader("QR_AUTHENTICATION_CHANNEL_ID", channel));
headers.add(new StompHeader("QR_AUTHENTICATION_DEVICE_ID", deviceId));
headers.add(new StompHeader(StompHeader.DESTINATION, "/qr/accept"));

// wss://10.0.2.2 for ssl and localhost
var client = Stomp.over(Stomp.ConnectionProvider.OKHTTP, 
  "wss://10.0.2.2:8443/cas/qr-websocket/websocket", null, httpClient);

client.connect();
var stompMessage = 
  new StompMessage(StompCommand.SEND, headers, jsonWebToken.toString());
client.send(stompMessage).subscribe();
```

## Obtaining JWT 

The mobile device should ask for and then submit user credentials to the CAS 
server using the [REST protocol](../protocol/REST-Protocol-Request-TicketGrantingTicket.html) to 
obtain a JWT. The JWT request must also contain an additional request 
parameter `QR_AUTHENTICATION_DEVICE_ID` which indicates the authorized device identifier for the user.

Once the JWT is received, the device may cache the JWT and establish a *session* for code reuse later. 
The JWT should be sent to the CAS server's web socket channel 
for validation and login as demonstrated above. The generated
JWT is automatically signed and encrypted by CAS and can only be decoded by the CAS server.

## Web Socket Channel

The QR code contains a special identifier embedded within that allows the mobile device to establish 
a communication channel using web sockets to the CAS server. The mobile device must be able 
to scan the QR code to extract the channel id in order to establish a 
communication route between CAS and the device.

## Mobile Device Authorization

Registered devices are authorized and accepted by CAS using a dedicated QR device repository, that is able to 
track and link device identifiers to user ids. Such devices must be registered with CAS using an 
external registration mechanism or via available CAS-provided APIs.

By default, all devices can authenticate using the QR code. Different device repository
implementations can be supplied using one of the strategies outlined below.

### JSON

Authorized devices can be managed and tracked inside a 
single JSON resource, whose path is taught to CAS via settings. 

{% include_cached casproperties.html properties="cas.authn.qr.json" %}

### Custom

Provide the appropriate bean implementation below to define a custom strategy for managing registered devices.

```java 
@Bean
public QRAuthenticationDeviceRepository qrAuthenticationDeviceRepository() {
    return ...
}
```
