package org.apereo.cas.nativex;

import module java.base;
import module java.sql;
import org.apereo.cas.configuration.support.TriStateBoolean;
import org.apereo.cas.util.CasVersion;
import org.apereo.cas.util.LogMessageSummarizer;
import org.apereo.cas.util.crypto.CipherExecutor;
import org.apereo.cas.util.function.FunctionUtils;
import org.apereo.cas.util.nativex.CasRuntimeHintsRegistrar;
import org.apereo.cas.util.serialization.ComponentSerializationPlanConfigurer;
import org.apereo.cas.util.serialization.MapContentDeserializer;
import org.apereo.cas.util.spring.RestActuatorEndpointFilter;
import org.apereo.cas.util.thread.Cleanable;
import com.fasterxml.jackson.annotation.ObjectIdGenerator;
import lombok.val;
import org.apache.commons.lang3.ClassUtils;
import org.jspecify.annotations.Nullable;
import org.slf4j.LoggerFactory;
import org.springframework.aot.hint.RuntimeHints;
import org.springframework.aot.hint.TypeReference;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.AutowiredAnnotationBeanPostProcessor;
import org.springframework.beans.factory.config.ListFactoryBean;
import org.springframework.beans.factory.config.SetFactoryBean;
import org.springframework.context.annotation.CommonAnnotationBeanPostProcessor;
import org.springframework.context.annotation.ConfigurationClassPostProcessor;
import org.springframework.context.event.DefaultEventListenerFactory;
import org.springframework.context.event.EventListenerMethodProcessor;
import org.springframework.orm.jpa.support.PersistenceAnnotationBeanPostProcessor;
import org.springframework.web.cors.CorsConfigurationSource;
import tools.jackson.databind.jsontype.NamedType;
import java.lang.module.Configuration;
import java.util.Date;

/**
 * This is {@link CasCoreUtilRuntimeHints}.
 *
 * @author Misagh Moayyed
 * @since 7.0.0
 */
public class CasCoreUtilRuntimeHints implements CasRuntimeHintsRegistrar {

    @Override
    public void registerHints(final RuntimeHints hints, final @Nullable ClassLoader classLoader) {
        hints.resources().registerType(CasVersion.class);

        registerProxyHints(hints, List.of(
            ComponentSerializationPlanConfigurer.class,
            InitializingBean.class,
            Supplier.class,
            Runnable.class,
            Function.class,
            Consumer.class,
            Cleanable.class,
            CorsConfigurationSource.class
        ));

        registerSerializationHints(hints);

        registerReflectionHintForDeclaredMethod(hints, Map.Entry.class, "getKey");
        registerReflectionHintForDeclaredMethod(hints, Map.Entry.class, "getValue");
        registerReflectionHintForDeclaredMethod(hints, Map.class, "isEmpty");

        registerReflectionHintsForMethodsAndFields(hints, List.of(
            BigDecimal.class,
            BigInteger.class,
            Math.class,
            URL.class,
            URI.class,
            SetFactoryBean.class,
            ListFactoryBean.class,
            CasVersion.class,
            Module.class,
            Class.class,
            Arrays.class,
            Collections.class,
            Collection.class,
            List.class,
            Iterator.class,
            Iterable.class,
            Queue.class,
            Set.class,
            Comparator.class,
            Comparable.class,
            ResultSet.class,
            Calendar.class,
            Date.class,
            SortedMap.class,
            SortedSet.class,
            TimeZone.class,
            BiPredicate.class,
            BiFunction.class,
            Predicate.class,
            Function.class,
            Consumer.class,
            Supplier.class,
            ModuleLayer.class,
            Configuration.class,
            ResolvedModule.class,
            ServiceLoader.class,
            Callable.class,
            Map.class,
            Locale.class
        ));

        registerReflectionHintsForPublicElements(hints,
            List.of(System.class, ExecutorService.class, Executor.class)
        );
        
        registerReflectionHintsForDeclaredElements(hints, List.of(
            HashMap.class,
            LinkedHashMap.class,
            TypeReference.of("java.time.Ser")
        ));

        registerReflectionHintsForPublicElements(hints, List.of(
            NamedType.class,
            TypeReference.of("java.util.LinkedHashMap$Entry"),
            TypeReference.of("java.util.TreeMap$Entry")
        ));

        registerReflectionHints(hints, List.of(
            ClassUtils.class,
            LoggerFactory.class
        ));

        registerReflectionHintsForConstructors(hints,
            List.of(
                MapContentDeserializer.class,
                TriStateBoolean.Deserializer.class,
                PersistenceAnnotationBeanPostProcessor.class,
                ConfigurationClassPostProcessor.class,
                EventListenerMethodProcessor.class,
                DefaultEventListenerFactory.class,
                AutowiredAnnotationBeanPostProcessor.class,
                CommonAnnotationBeanPostProcessor.class,
                RestActuatorEndpointFilter.class
            ));

        registerReflectionHintsForTypes(hints,
            List.of(
                TypeReference.of("java.util.HashMap$Node"),
                TypeReference.of("java.util.HashMap$TreeNode"))
        );

        registerReflectionHintsForTypes(hints, findSubclassesInPackage(Clock.class, Clock.class.getPackageName()));
        registerReflectionHintsForPublicElements(hints, findSubclassesInPackage(ObjectIdGenerator.class, "com.fasterxml.jackson"));
        registerReflectionHintsForPublicElements(hints, findSubclassesInPackage(LogMessageSummarizer.class, "org.apereo.cas"));
        registerReflectionHintsForPublicElements(hints, findSubclassesInPackage(CipherExecutor.class, "org.apereo.cas"));

        registerCaffeineHints(hints);
        
        FunctionUtils.doAndHandle(_ -> {
            val clazz = ClassUtils.getClass("nonapi.io.github.classgraph.classloaderhandler.ClassLoaderHandler", false);
            registerReflectionHints(hints, findSubclassesInPackage(clazz, "nonapi.io.github.classgraph.classloaderhandler"));
        });
    }
    
    private void registerCaffeineHints(final RuntimeHints hints) {
        FunctionUtils.doAndHandle(_ -> {
            var clazz = ClassUtils.getClass("com.github.benmanes.caffeine.cache.Node", false);
            registerReflectionHintsForDeclaredAndPublicElements(hints, findSubclassesInPackage(clazz, "com.github.benmanes.caffeine.cache"));
            clazz = ClassUtils.getClass("com.github.benmanes.caffeine.cache.LocalCache", false);
            registerReflectionHintsForDeclaredAndPublicElements(hints, findSubclassesInPackage(clazz, "com.github.benmanes.caffeine.cache"));
        });
    }

    private void registerSerializationHints(final RuntimeHints hints) {
        registerSerializationHints(hints,
            Boolean.class,
            Double.class,
            Integer.class,
            Long.class,
            String.class,
            Float.class,

            ZonedDateTime.class,
            LocalDateTime.class,
            LocalDate.class,
            LocalTime.class,
            ZoneId.class,
            ZoneOffset.class,
            Instant.class,
            Locale.class,
            
            ArrayList.class,
            Vector.class,
            CopyOnWriteArrayList.class,
            LinkedList.class,

            HashMap.class,
            LinkedHashMap.class,
            ConcurrentHashMap.class,
            TreeMap.class,

            ConcurrentSkipListSet.class,
            ConcurrentLinkedQueue.class,
            ConcurrentHashMap.class,
            HashSet.class,
            LinkedHashSet.class,
            CopyOnWriteArraySet.class,
            TreeSet.class,

            TypeReference.of("java.lang.String$CaseInsensitiveComparator"));

        registerSerializationHints(hints, findSubclassesInPackage(Clock.class, Clock.class.getPackageName()));
    }
}
