package org.apereo.cas.util.spring;

import module java.base;
import lombok.Setter;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.NoSuchMessageException;
import org.springframework.context.i18n.LocaleContextHolder;
import jakarta.validation.MessageInterpolator;
import jakarta.validation.Validation;

/**
 * Configures the {@link jakarta.validation.Validator} to check the Spring Messages.
 *
 * @author Scott Battaglia
 * @since 3.4
 */
@Setter
public class SpringAwareMessageMessageInterpolator implements MessageInterpolator, MessageSourceAware {

    private final MessageInterpolator defaultMessageInterpolator = Validation.byDefaultProvider().configure().getDefaultMessageInterpolator();

    private MessageSource messageSource;

    @Override
    public String interpolate(final String s, final Context context) {
        return interpolate(s, context, LocaleContextHolder.getLocale());
    }

    @Override
    public String interpolate(final String s, final Context context, final Locale locale) {
        try {
            return this.messageSource.getMessage(s, context.getConstraintDescriptor()
                .getAttributes().values().toArray(ArrayUtils.EMPTY_OBJECT_ARRAY), locale);
        } catch (final NoSuchMessageException e) {
            return this.defaultMessageInterpolator.interpolate(s, context, locale);
        }
    }
}
