package org.apereo.cas.mock;

import module java.base;
import org.apereo.cas.authentication.Authentication;
import org.apereo.cas.authentication.principal.Service;
import org.apereo.cas.ticket.ExpirationPolicy;
import org.apereo.cas.ticket.ProxyGrantingTicketImpl;
import org.apereo.cas.ticket.ProxyGrantingTicketIssuerTicket;
import org.apereo.cas.ticket.RenewableServiceTicket;
import org.apereo.cas.ticket.ServiceTicket;
import org.apereo.cas.ticket.Ticket;
import org.apereo.cas.ticket.TicketGrantingTicket;
import org.apereo.cas.ticket.expiration.NeverExpiresExpirationPolicy;
import org.apereo.cas.ticket.proxy.ProxyGrantingTicket;
import org.apereo.cas.ticket.tracking.TicketTrackingPolicy;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.val;

/**
 * Mock service ticket.
 *
 * @author Marvin S. Addison
 * @since 3.0.0
 */
@Getter
@Setter
@EqualsAndHashCode(of = "id")
public class MockServiceTicket implements ServiceTicket, RenewableServiceTicket, ProxyGrantingTicketIssuerTicket {

    @Serial
    private static final long serialVersionUID = 8203377063087967768L;

    private final String id;
    
    private String tenantId;

    private final ZonedDateTime creationTime;

    private final Service service;

    private TicketGrantingTicket ticketGrantingTicket;

    private ExpirationPolicy expirationPolicy;

    private boolean expired;

    public MockServiceTicket(final String id, final Service service,
                             final TicketGrantingTicket parent) {
        this(id, service, parent, NeverExpiresExpirationPolicy.INSTANCE);
    }

    public MockServiceTicket(final String id,
                             final Service service,
                             final TicketGrantingTicket parent,
                             final ExpirationPolicy policy) {
        this.service = service;
        this.id = id;
        this.ticketGrantingTicket = parent;
        this.creationTime = ZonedDateTime.now(ZoneOffset.UTC);
        this.expirationPolicy = Optional.ofNullable(policy).orElse(NeverExpiresExpirationPolicy.INSTANCE);
    }

    @Override
    public ProxyGrantingTicket grantProxyGrantingTicket(final String id, final Authentication authentication,
                                                        final ExpirationPolicy expirationPolicy,
                                                        final TicketTrackingPolicy proxyGrantingTicketTrackingPolicy) {
        val pgt = new ProxyGrantingTicketImpl(id, this.service, this.getTicketGrantingTicket(), authentication, expirationPolicy);
        proxyGrantingTicketTrackingPolicy.trackTicket(getTicketGrantingTicket(), pgt, this.service);
        return pgt;
    }

    @Override
    public Authentication getAuthentication() {
        return this.ticketGrantingTicket.getAuthentication();
    }

    @Override
    public int getCountOfUses() {
        return 0;
    }

    @Override
    public String getPrefix() {
        return ServiceTicket.PREFIX;
    }

    @Override
    public boolean isExpired() {
        return this.expired || this.expirationPolicy.isExpired(this);
    }

    @Override
    public void markTicketExpired() {
        this.expired = true;
    }

    @Override
    public ZonedDateTime getLastTimeUsed() {
        return ZonedDateTime.now(ZoneId.systemDefault());
    }

    @Override
    public ZonedDateTime getPreviousTimeUsed() {
        return ZonedDateTime.now(ZoneId.systemDefault());
    }

    @Override
    public int compareTo(final Ticket o) {
        return this.id.compareTo(o.getId());
    }

    @Override
    public boolean isFromNewLogin() {
        return false;
    }
}
