package org.apereo.cas.authentication.principal;

import module java.base;
import org.apereo.cas.CasProtocolConstants;
import org.apereo.cas.multitenancy.TenantExtractor;
import org.apereo.cas.util.CollectionUtils;
import org.apereo.cas.util.http.HttpRequestUtils;
import org.apereo.cas.validation.ValidationResponseType;
import org.apereo.cas.web.UrlValidator;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.jspecify.annotations.Nullable;
import jakarta.servlet.http.HttpServletRequest;

/**
 * The {@link WebApplicationServiceFactory} is responsible for
 * creating {@link WebApplicationService} objects.
 *
 * @author Misagh Moayyed
 * @since 4.2
 */
@Slf4j
public class WebApplicationServiceFactory extends AbstractServiceFactory<WebApplicationService> {

    public WebApplicationServiceFactory(final TenantExtractor tenantExtractor,
                                        final UrlValidator urlValidator) {
        super(tenantExtractor, urlValidator);
    }

    private static AbstractWebApplicationService determineWebApplicationFormat(
        @Nullable final HttpServletRequest request,
        final AbstractWebApplicationService webApplicationService) {
        val format = Optional.ofNullable(request)
            .map(httpServletRequest -> httpServletRequest.getParameter(CasProtocolConstants.PARAMETER_FORMAT))
            .orElse(StringUtils.EMPTY);
        try {
            if (StringUtils.isNotBlank(format)) {
                val formatType = ValidationResponseType.valueOf(Objects.requireNonNull(format).toUpperCase(Locale.ENGLISH));
                webApplicationService.setFormat(formatType);
            }
        } catch (final Exception e) {
            LOGGER.error("Format specified in the request [{}] is not recognized", format);
        }
        return webApplicationService;
    }

    @Override
    public @Nullable WebApplicationService createService(@Nullable final HttpServletRequest request) {
        val serviceToUse = getRequestedService(request);
        if (StringUtils.isBlank(serviceToUse)) {
            LOGGER.trace("No service is specified in the request. Skipping service creation");
            return null;
        }
        return newWebApplicationService(request, serviceToUse);
    }

    @Override
    public WebApplicationService createService(final String id) {
        val request = HttpRequestUtils.getHttpServletRequestFromRequestAttributes();
        return newWebApplicationService(request, id);
    }

    protected WebApplicationService newWebApplicationService(
        @Nullable final HttpServletRequest request, final String serviceToUse) {
        val artifactId = Optional.ofNullable(request)
            .map(httpServletRequest -> httpServletRequest.getParameter(CasProtocolConstants.PARAMETER_TICKET))
            .orElse(null);
        val id = cleanupUrl(serviceToUse);
        val newService = new SimpleWebApplicationServiceImpl(id, serviceToUse, artifactId);
        determineWebApplicationFormat(request, newService);
        val source = getSourceParameter(request, CasProtocolConstants.PARAMETER_TARGET_SERVICE, CasProtocolConstants.PARAMETER_SERVICE);
        newService.setSource(source);
        if (request != null) {
            populateAttributes(newService, request);
            if (StringUtils.isNotBlank(source)) {
                newService.getAttributes().put(source, CollectionUtils.wrap(id));
            }
        } else {
            newService.getAttributes().putAll(extractQueryParameters(newService));
        }
        return newService;
    }

    protected @Nullable String getRequestedService(@Nullable final HttpServletRequest request) {
        if (request != null) {
            val targetService = request.getParameter(CasProtocolConstants.PARAMETER_TARGET_SERVICE);
            val service = request.getParameter(CasProtocolConstants.PARAMETER_SERVICE);
            val serviceAttribute = request.getAttribute(CasProtocolConstants.PARAMETER_SERVICE);

            if (StringUtils.isNotBlank(targetService)) {
                return targetService;
            }
            if (StringUtils.isNotBlank(service)) {
                return service;
            }
            if (serviceAttribute != null) {
                if (serviceAttribute instanceof final Service svc) {
                    return svc.getId();
                }
                return serviceAttribute.toString();
            }
        }
        return null;
    }
}
