package org.apereo.cas.authentication.handler;

import module java.base;
import org.apereo.cas.authentication.AuthenticationHandler;
import org.apereo.cas.authentication.AuthenticationHandlerResolver;
import org.apereo.cas.authentication.AuthenticationServiceSelectionPlan;
import org.apereo.cas.authentication.AuthenticationTransaction;
import org.apereo.cas.authentication.MultifactorAuthenticationHandler;
import org.apereo.cas.authentication.handler.support.ProxyAuthenticationHandler;
import org.apereo.cas.authentication.principal.Service;
import org.apereo.cas.services.RegisteredService;
import org.apereo.cas.services.ServicesManager;
import org.apereo.cas.services.UnauthorizedSsoServiceException;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.jspecify.annotations.Nullable;

/**
 * This is {@link RegisteredServiceAuthenticationHandlerResolver}
 * that acts on the criteria presented by a registered service to
 * detect which handler(s) should be resolved for authentication.
 *
 * @author Misagh Moayyed
 * @since 5.0.0
 */
@Slf4j
@RequiredArgsConstructor
@Getter
@Setter
public class RegisteredServiceAuthenticationHandlerResolver implements AuthenticationHandlerResolver {

    /**
     * The Services manager.
     */
    protected final ServicesManager servicesManager;

    /**
     * The service selection plan.
     */
    protected final AuthenticationServiceSelectionPlan authenticationServiceSelectionPlan;

    private int order;

    @Override
    public Set<AuthenticationHandler> resolve(final Set<AuthenticationHandler> candidateHandlers,
                                              final AuthenticationTransaction transaction) throws Throwable {
        val service = authenticationServiceSelectionPlan.resolveService(transaction.getService());
        val registeredService = servicesManager.findServiceBy(service);

        val requiredHandlers = filterRequiredAuthenticationHandlers(candidateHandlers, service, registeredService, transaction);
        return filterExcludedAuthenticationHandlers(requiredHandlers, service, registeredService);
    }

    @Override
    public boolean supports(final Set<AuthenticationHandler> handlers, final AuthenticationTransaction transaction) throws Throwable {
        val service = authenticationServiceSelectionPlan.resolveService(transaction.getService());
        if (service != null) {
            val registeredService = servicesManager.findServiceBy(service);
            LOGGER.trace("Located registered service definition [{}] for this authentication transaction", registeredService);
            if (registeredService == null || !registeredService.getAccessStrategy().isServiceAccessAllowed(registeredService, service)) {
                LOGGER.warn("Service [{}] is not allowed to use SSO.", service);
                throw new UnauthorizedSsoServiceException("Denied: %s".formatted(service));
            }
            val authenticationPolicy = registeredService.getAuthenticationPolicy();
            return !authenticationPolicy.getRequiredAuthenticationHandlers().isEmpty()
                   || !authenticationPolicy.getExcludedAuthenticationHandlers().isEmpty();
        }
        return false;
    }

    protected Set<AuthenticationHandler> filterExcludedAuthenticationHandlers(
        final Set<AuthenticationHandler> candidateHandlers,
        @Nullable final Service service,
        @Nullable final RegisteredService registeredService) {

        val authenticationPolicy = Objects.requireNonNull(registeredService).getAuthenticationPolicy();
        val excludedHandlers = authenticationPolicy.getExcludedAuthenticationHandlers();
        LOGGER.debug("Authentication transaction excludes [{}] for service [{}]", excludedHandlers, service);

        val handlerSet = new LinkedHashSet<>(candidateHandlers);
        LOGGER.debug("Candidate authentication handlers examined for exclusion in this transaction are [{}]", handlerSet);

        if (!excludedHandlers.isEmpty()) {
            val it = handlerSet.iterator();
            while (it.hasNext()) {
                val handler = it.next();
                val handlerName = handler.getName();
                if (excludedHandlers.contains(handlerName)) {
                    LOGGER.debug("Authentication handler [{}] is excluded for this transaction and is removed", handlerName);
                    it.remove();
                }
            }
        }
        LOGGER.info("Final authentication handlers after exclusion rules are [{}]", handlerSet);
        return handlerSet;
    }

    protected Set<AuthenticationHandler> filterRequiredAuthenticationHandlers(
        final Set<AuthenticationHandler> candidateHandlers,
        @Nullable final Service service, @Nullable final RegisteredService registeredService,
        final AuthenticationTransaction transaction) {

        val authenticationPolicy = Objects.requireNonNull(registeredService).getAuthenticationPolicy();
        val requiredHandlers = authenticationPolicy.getRequiredAuthenticationHandlers();
        LOGGER.debug("Authentication transaction requires [{}] for service [{}]", requiredHandlers, service);
        val handlerSet = new LinkedHashSet<>(candidateHandlers);
        LOGGER.debug("Candidate authentication handlers examined for this transaction are [{}]", handlerSet);

        if (!requiredHandlers.isEmpty()) {
            val it = handlerSet.iterator();
            while (it.hasNext()) {
                val handler = it.next();
                val handlerName = handler.getName();
                val removeHandler = !(handler instanceof MultifactorAuthenticationHandler)
                    && !(handler instanceof ProxyAuthenticationHandler)
                    && !requiredHandlers.contains(handlerName);
                if (removeHandler) {
                    it.remove();
                    LOGGER.debug("Authentication handler [{}] is removed", handlerName);
                }
            }
        }
        LOGGER.info("Final authentication handlers after inclusion rules are [{}]", handlerSet);
        return handlerSet;
    }
}
