package org.apereo.cas.util.serialization;

import module java.base;
import lombok.val;
import org.springframework.http.MediaType;

/**
 * Interface to define operations needed to map objects from/to  clobs.
 *
 * @author Misagh Moayyed
 * @param <T> the type parameter
 * @since 4.1.0
 */
public interface StringSerializer<T> extends Serializable {
    /**
     * Helper method with reusable code.
     *
     * @param <T>  the type param
     * @param elem the elem
     * @return collection
     */
    private static <T> Collection<T> makeCollectionOf(final T elem) {
        if (elem != null) {
            val list = new ArrayList<T>(1);
            list.add(elem);
            return list;
        }
        return new ArrayList<>();
    }

    /**
     * Create the object type from the given  string.
     *
     * @param data the  string
     * @return the object instance constructed from
     */
    T from(String data);

    /**
     * Create the object type from the given  reader.
     *
     * @param reader the  string
     * @return the object instance constructed from
     */
    T from(Reader reader);

    /**
     * Create the object type from the given  stream.
     *
     * @param stream the  string
     * @return the object instance constructed from
     */
    T from(InputStream stream);

    /**
     * Create the object type from the given  file.
     *
     * @param file the  string
     * @return the object instance constructed from
     */
    T from(File file);

    /**
     * Create the object type from the given writer.
     *
     * @param writer the writer instance
     * @return the object instance constructed from
     */
    T from(Writer writer);

    /**
     * Serialize the given object to its  equivalent to the output stream.
     *
     * @param out    the output stream
     * @param object the object to serialize
     */
    void to(OutputStream out, T object);

    /**
     * Serialize the given object to its  equivalent to the output writer.
     *
     * @param out    the output writer
     * @param object the object to serialize
     */
    void to(Writer out, T object);

    /**
     * Serialize the given object to its  equivalent to the output file.
     *
     * @param out    the output file
     * @param object the object to serialize
     */
    void to(File out, T object);

    /**
     * Return the object as a string in memory.
     *
     * @param object the object
     * @return string representation of the object.
     */
    String toString(T object);

    /**
     * Load a collection of specified objects from the stream.
     *
     * @param stream the stream
     * @return the collection
     */
    default Collection<T> load(final InputStream stream) {
        val result = from(stream);
        return makeCollectionOf(result);
    }

    /**
     * Load collection.
     *
     * @param stream the stream
     * @return the collection
     */
    default Collection<T> load(final Reader stream) {
        val result = from(stream);
        return makeCollectionOf(result);
    }

    /**
     * Supports the input stream for serialization?
     *
     * @param file the file
     * @return true /false
     */
    default boolean supports(final File file) {
        return true;
    }

    /**
     * Supports the input stream for serialization?
     *
     * @param content the content
     * @return true /false
     */
    default boolean supports(final String content) {
        return true;
    }

    /**
     * Gets type to serialize.
     *
     * @return the type to serialize
     */
    Class<T> getTypeToSerialize();

    /**
     * Read and consume the input
     * and parse the result into a list/collection.
     *
     * @param json the json
     * @return the list
     */
    List<T> fromList(String json);

    /**
     * Collect a list of objects and transform them into a String
     * in the intended format.
     *
     * @param json the json
     * @return the string
     */
    String fromList(Collection<T> json);
    
    default List<MediaType> getContentTypes() {
        return List.of(MediaType.TEXT_PLAIN);
    }

    /**
     * Merge.
     *
     * @param baseEntity  the base entity
     * @param childEntity the child entity
     * @return the result
     */
    T merge(T baseEntity, T childEntity);
}
