package org.apereo.cas.ticket;

import module java.base;
import org.apereo.cas.authentication.Authentication;
import org.apereo.cas.authentication.principal.Service;
import org.apereo.cas.ticket.tracking.TicketTrackingPolicy;
import com.fasterxml.jackson.annotation.JsonTypeInfo;

/**
 * Interface for a ticket granting ticket. A TicketGrantingTicket is the main
 * access into the CAS service layer. Without a TicketGrantingTicket, a user of
 * CAS cannot do anything.
 *
 * @author Scott Battaglia
 * @since 3.0.0
 */
@JsonTypeInfo(use = JsonTypeInfo.Id.CLASS)
public interface TicketGrantingTicket extends TicketGrantingTicketAwareTicket {

    /**
     * The prefix to use when generating an id for a Ticket Granting Ticket.
     */
    String PREFIX = "TGT";

    /**
     * Grant a ServiceTicket for a specific service.
     * <p>The state of the ticket is affected by this operation and the
     * ticket will be considered used. The state update subsequently may
     * impact the ticket expiration policy in that, depending on the policy
     * configuration, the ticket may be considered expired.
     *
     * @param id                 The unique identifier for this ticket.
     * @param service            The service for which we are granting a ticket
     * @param expirationPolicy   the expiration policy.
     * @param credentialProvided current credential event for issuing this ticket. Could be null.
     * @param trackingPolicy     the tracking policy
     * @return the service ticket granted to a specific service for the principal of the TicketGrantingTicket
     */
    ServiceTicket grantServiceTicket(String id,
                                     Service service,
                                     ExpirationPolicy expirationPolicy,
                                     boolean credentialProvided,
                                     TicketTrackingPolicy trackingPolicy);

    /**
     * Gets proxy granting tickets created by this TGT.
     *
     * @return the proxy granting tickets
     */
    Map<String, Service> getProxyGrantingTickets();

    /**
     * Remove all services of the TGT (at logout).
     */
    void removeAllServices();

    /**
     * Convenience method to determine if the TicketGrantingTicket is the root
     * of the hierarchy of tickets.
     *
     * @return true if it has no parent, false otherwise.
     */
    boolean isRoot();

    /**
     * Gets the ticket-granting ticket at the root of the ticket hierarchy.
     *
     * @return Non -null root ticket-granting ticket.
     */
    TicketGrantingTicket getRoot();

    /**
     * Gets all authentications ({@link AuthenticationAwareTicket#getAuthentication()} from this
     * instance and all dependent tickets that reference this one.
     *
     * @return Non -null list of authentication associated with this ticket in leaf-first order.
     */
    List<Authentication> getChainedAuthentications();

    /**
     * Gets the service that produced a proxy-granting ticket.
     *
     * @return Service that produced proxy-granting ticket or null if this is not a proxy-granting ticket.
     * @since 4.1
     */
    Service getProxiedBy();

    /**
     * Gets descendant tickets. These are generally ticket ids
     * whose life-line is separate from the TGT until and unless
     * the TGT goes away entirely. Things such as OAuth access tokens
     * are a good example of such linked tickets.
     *
     * @return the descendant tickets
     * @since 5.1
     */
    default Collection<String> getDescendantTickets() {
        return new HashSet<>();
    }

    /**
     * Keeps track of authenticated service per their session id.
     *
     * @return authenticated services.
     */
    Map<String, Service> getServices();
}
