package org.apereo.cas.services;

import module java.base;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.val;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.jspecify.annotations.Nullable;

/**
 * The {@link RegisteredServiceProperty} defines a single custom
 * property that is associated with a service.
 *
 * @author Misagh Moayyed
 * @since 4.2
 */
@JsonTypeInfo(use = JsonTypeInfo.Id.CLASS, include = JsonTypeInfo.As.PROPERTY)
public interface RegisteredServiceProperty extends Serializable {

    /**
     * Gets values.
     *
     * @return the values
     */
    Set<String> getValues();

    /**
     * Gets the first single value.
     *
     * @return the value, or null if the collection is empty.
     */
    @JsonIgnore
    String value();

    /**
     * Gets property value.
     *
     * @param <T>   the type parameter
     * @param clazz the clazz
     * @return the property value
     */
    @JsonIgnore
    default <T> T getValue(final Class<T> clazz) {
        val value = value();
        if (StringUtils.isNotBlank(value)) {
            return clazz.cast(value);
        }
        return null;
    }

    /**
     * Contains elements?
     *
     * @param value the value
     * @return true/false
     */
    boolean contains(String value);

    /**
     * Gets property value.
     *
     * @return the property value
     */
    @JsonIgnore
    default boolean getBooleanValue() {
        val value = value();
        return StringUtils.isNotBlank(value) && BooleanUtils.toBoolean(value);
    }

    /**
     * Indicates the group for each property.
     */
    @JsonFormat(shape = JsonFormat.Shape.OBJECT)
    @Getter
    @RequiredArgsConstructor
    enum RegisteredServicePropertyGroups {
        /**
         * Property grouup for OpenID Connect.
         */
        OIDC,
        /**
         * Property group for CORS settings.
         */
        CORS,
        /**
         * Property group for delegated authn settings.
         */
        DELEGATED_AUTHN,
        /**
         * Property group for SAML2 delegated authn settings.
         */
        DELEGATED_AUTHN_SAML2,
        /**
         * Property group for WSFED authn settings.
         */
        DELEGATED_AUTHN_WSFED,
        /**
         * Property group for OIDC delegated authn settings.
         */
        DELEGATED_AUTHN_OIDC,
        /**
         * Property group for http headers settings.
         */
        HTTP_HEADERS,
        /**
         * Property group for interrupt settings.
         */
        INTERRUPTS,
        /**
         * Property group for JWT authn settings.
         */
        JWT_AUTHENTICATION,
        /**
         * Property group for JWT access token settings.
         */
        JWT_ACCESS_TOKENS,
        /**
         * Property group for JWT tokens settings.
         */
        JWT_TOKENS,
        /**
         * Property group for JWT service ticket settings.
         */
        JWT_SERVICE_TICKETS,
        /**
         * Property group for registered services settings.
         */
        REGISTERED_SERVICES,
        /**
         * Property group for reCAPTCHA settings.
         */
        RECAPTCHA,
        /**
         * Property group for SCIM settings.
         */
        SCIM
    }

    /**
     * Indicates the property type for each property.
     */
    @JsonFormat(shape = JsonFormat.Shape.OBJECT)
    @Getter
    @RequiredArgsConstructor
    enum RegisteredServicePropertyTypes {
        /**
         * Property type indicates a set or collection.
         */
        SET,
        /**
         * Property type indicates a string.
         */
        STRING,
        /**
         * Property type indicates a integer.
         */
        INTEGER,
        /**
         * Property type indicates a boolean.
         */
        BOOLEAN,
        /**
         * Property type indicates a long value.
         */
        LONG
    }

    /**
     * Collection of supported properties that
     * control various functionality in CAS.
     */
    @JsonFormat(shape = JsonFormat.Shape.OBJECT)
    @Getter
    @RequiredArgsConstructor
    enum RegisteredServiceProperties {
        /**
         * Used when delegating authentication to ADFS to indicate the relying party identifier.
         */
        WSFED_RELYING_PARTY_ID("wsfed.relyingPartyIdentifier", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_WSFED, RegisteredServicePropertyTypes.STRING,
            "Used when delegating authentication to ADFS to indicate the relying party identifier."),
        /**
         * Produce a JWT as a response when generating service tickets.
         **/
        TOKEN_AS_SERVICE_TICKET("jwtAsServiceTicket", "false",
            RegisteredServicePropertyGroups.JWT_SERVICE_TICKETS, RegisteredServicePropertyTypes.BOOLEAN,
            "Produce a JWT as a response when generating service tickets."),
        /**
         * Indicate the cipher strategy for JWT service tickets to determine order of signing/encryption operations.
         **/
        TOKEN_AS_SERVICE_TICKET_CIPHER_STRATEGY_TYPE("jwtAsServiceTicketCipherStrategyType", "ENCRYPT_AND_SIGN",
            RegisteredServicePropertyGroups.JWT_SERVICE_TICKETS, RegisteredServicePropertyTypes.STRING,
            "Indicate the cipher strategy for JWT service tickets to determine order of signing/encryption operations."),
        /**
         * Produce a signed JWT as a response when generating service tickets using the provided signing key.
         **/
        TOKEN_AS_SERVICE_TICKET_SIGNING_KEY("jwtAsServiceTicketSigningKey", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_SERVICE_TICKETS, RegisteredServicePropertyTypes.STRING,
            "Produce a signed JWT as a response when generating service tickets using the provided signing key."),
        /**
         * Produce an encrypted JWT as a response when generating service tickets using the provided encryption key.
         **/
        TOKEN_AS_SERVICE_TICKET_ENCRYPTION_KEY("jwtAsServiceTicketEncryptionKey", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_SERVICE_TICKETS, RegisteredServicePropertyTypes.STRING,
            "Produce an encrypted JWT as a response when generating service tickets using the provided encryption key."),
        /**
         * Encryption algorithm when generating service tickets using the provided encryption key.
         **/
        TOKEN_AS_SERVICE_TICKET_ENCRYPTION_ALG("jwtAsServiceTicketEncryptionAlg", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_SERVICE_TICKETS, RegisteredServicePropertyTypes.STRING,
            "Encryption algorithm when generating service tickets using the provided encryption key."),
        /**
         * Whether signing operations should be enabled when producing JWTs.
         **/
        TOKEN_AS_SERVICE_TICKET_SIGNING_ENABLED("jwtAsServiceTicketSigningEnabled", "true",
            RegisteredServicePropertyGroups.JWT_SERVICE_TICKETS, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether signing operations should be enabled when producing JWTs."),

        /**
         * Whether encryption operations should be enabled when producing JWTs.
         **/
        TOKEN_AS_SERVICE_TICKET_ENCRYPTION_ENABLED("jwtAsServiceTicketEncryptionEnabled", "true",
            RegisteredServicePropertyGroups.JWT_SERVICE_TICKETS, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether encryption operations should be enabled when producing JWTs."),

        /**
         * Indicate whether the client was registered with CAS using OpenID Connect client dynamic registration flow.
         */
        OIDC_DYNAMIC_CLIENT_REGISTRATION("oidcDynamicClientRegistration", "false",
            RegisteredServicePropertyGroups.OIDC, RegisteredServicePropertyTypes.BOOLEAN,
            "Indicate whether the client was registered with CAS using OpenID Connect client dynamic registration flow."),

        /**
         * Indicate the registration date/time when the client was registered with CAS using OpenID Connect client dynamic registration flow.
         */
        OIDC_DYNAMIC_CLIENT_REGISTRATION_DATE("oidcDynamicClientRegistrationDate", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.OIDC, RegisteredServicePropertyTypes.STRING,
            "Indicate the registration date/time when the client was registered with CAS using OpenID Connect client dynamic registration flow."),

        /**
         * Indicate the cipher strategy for JWTs for OIDC responses, to determine order of signing/encryption operations.
         */
        OIDC_RESPONSE_MODE_JWT_CIPHER_STRATEGY_TYPE("oidcResponseModeAsJwtCipherStrategyType", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.OIDC, RegisteredServicePropertyTypes.STRING,
            "Indicate the cipher strategy for JWTs for OIDC responses, to determine order of signing/encryption operations."),

        /**
         * Enable signing JWTs as a response when generating response mode JWTs using the provided signing key.
         **/
        OIDC_RESPONSE_MODE_JWT_CIPHER_SIGNING_ENABLED("oidcResponseModeAsJwtCipherSigningEnabled", "true",
            RegisteredServicePropertyGroups.JWT_ACCESS_TOKENS, RegisteredServicePropertyTypes.BOOLEAN,
            "Enable signing JWTs as a response when generating response mode JWTs using the provided signing key."),

        /**
         * Enable encrypted JWTs as a response when generating response mode JWTs using the provided signing key.
         **/
        OIDC_RESPONSE_MODE_JWT_CIPHER_ENCRYPTION_ENABLED("oidcResponseModeAsJwtCipherEncryptionEnabled", "true",
            RegisteredServicePropertyGroups.JWT_ACCESS_TOKENS, RegisteredServicePropertyTypes.BOOLEAN,
            "Enable encrypted JWTs as a response when generating response mode JWTs using the provided encryption key."),

        /**
         * Produce a signed JWT as a response when generating access tokens using the provided signing key.
         **/
        ACCESS_TOKEN_AS_JWT_SIGNING_KEY("accessTokenAsJwtSigningKey", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_ACCESS_TOKENS, RegisteredServicePropertyTypes.STRING,
            "Produce a signed JWT as a response when generating access tokens using the provided signing key."),
        /**
         * Indicate the cipher strategy for JWTs as access tokens, to determine order of signing/encryption operations.
         */
        ACCESS_TOKEN_AS_JWT_CIPHER_STRATEGY_TYPE("accessTokenAsJwtCipherStrategyType", "ENCRYPT_AND_SIGN",
            RegisteredServicePropertyGroups.JWT_ACCESS_TOKENS, RegisteredServicePropertyTypes.STRING,
            "Indicate the cipher strategy for JWTs as access tokens, to determine order of signing/encryption operations."),
        /**
         * Enable signing JWTs as a response when generating access tokens using the provided signing key.
         **/
        ACCESS_TOKEN_AS_JWT_SIGNING_ENABLED("accessTokenAsJwtSigningEnabled", "true",
            RegisteredServicePropertyGroups.JWT_ACCESS_TOKENS, RegisteredServicePropertyTypes.BOOLEAN,
            "Enable signing JWTs as a response when generating access tokens using the provided signing key."),
        /**
         * Enable encryption of JWTs as a response when generating access tokens using the provided encryption key.
         **/
        ACCESS_TOKEN_AS_JWT_ENCRYPTION_ENABLED("accessTokenAsJwtEncryptionEnabled", "false",
            RegisteredServicePropertyGroups.JWT_ACCESS_TOKENS, RegisteredServicePropertyTypes.BOOLEAN,
            "Enable encryption of JWTs as a response when generating access tokens using the provided encryption key."),
        /**
         * Produce an encrypted JWT as a response when generating access tokens using the provided encryption key.
         **/
        ACCESS_TOKEN_AS_JWT_ENCRYPTION_KEY("accessTokenAsJwtEncryptionKey", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_ACCESS_TOKENS, RegisteredServicePropertyTypes.STRING,
            "Produce an encrypted JWT as a response when generating access tokens using the provided encryption key."),
        /**
         * Encryption algorithm to use when generating access tokens using the provided encryption key.
         */
        ACCESS_TOKEN_AS_JWT_ENCRYPTION_ALG("accessTokenAsJwtEncryptionAlg", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_ACCESS_TOKENS, RegisteredServicePropertyTypes.STRING,
            "Encryption algorithm to use when generating access tokens using the provided encryption key."),
        /**
         * Jwt signing secret defined for a given service.
         **/
        TOKEN_SECRET_SIGNING("jwtSigningSecret", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_AUTHENTICATION, RegisteredServicePropertyTypes.STRING,
            "Jwt signing secret defined for a given service. This property supports the spring expression language and may point to an external private key file."),
        /**
         * Jwt signing secret alg defined for a given service.
         **/
        TOKEN_SECRET_SIGNING_ALG("jwtSigningSecretAlg", "HS256",
            RegisteredServicePropertyGroups.JWT_AUTHENTICATION, RegisteredServicePropertyTypes.STRING,
            "Jwt signing secret alg defined for a given service."),
        /**
         * Jwt encryption secret defined for a given service.
         **/
        TOKEN_SECRET_ENCRYPTION("jwtEncryptionSecret", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_AUTHENTICATION, RegisteredServicePropertyTypes.STRING,
            "Jwt encryption secret defined for a given service. This property supports the spring expression language and may point to an external public key file."),
        /**
         * Jwt encryption secret alg defined for a given service.
         **/
        TOKEN_SECRET_ENCRYPTION_ALG("jwtEncryptionSecretAlg", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.JWT_AUTHENTICATION, RegisteredServicePropertyTypes.STRING,
            "Jwt encryption secret alg defined for a given service."),
        /**
         * Jwt encryption secret method defined for a given service.
         **/
        TOKEN_SECRET_ENCRYPTION_METHOD("jwtEncryptionSecretMethod", "A192CBC-HS384",
            RegisteredServicePropertyGroups.JWT_AUTHENTICATION, RegisteredServicePropertyTypes.STRING,
            "Jwt encryption secret method defined for a given service."),
        /**
         * Secrets are Base64 encoded.
         **/
        TOKEN_SECRETS_ARE_BASE64_ENCODED("jwtSecretsAreBase64Encoded", "false",
            RegisteredServicePropertyGroups.JWT_AUTHENTICATION, RegisteredServicePropertyTypes.BOOLEAN,
            "Determine whether secrets are Base64 encoded."),
        /**
         * Whether this service definition is one that is tagged as wildcarded (catch-all) entry.
         **/
        WILDCARDED_SERVICE_DEFINITION("wildcardedServiceDefinition", "false",
            RegisteredServicePropertyGroups.REGISTERED_SERVICES, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether this service definition is one that is tagged as wildcarded (catch-all) entry."),
        /**
         * Whether this service definition is one that is tagged for internal CAS functions.
         **/
        INTERNAL_SERVICE_DEFINITION("internalServiceDefinition", "false",
            RegisteredServicePropertyGroups.REGISTERED_SERVICES, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether this service definition is an internal system-level entry used by CAS itself."),
        /**
         * Whether this service should skip qualification for required-service pattern checks.
         **/
        SKIP_REQUIRED_SERVICE_CHECK("skipRequiredServiceCheck", "false",
            RegisteredServicePropertyGroups.REGISTERED_SERVICES, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether this service should skip qualification for required-service pattern checks."),
        /**
         * Whether CAS should inject cache control headers into the response when this service is in process.
         */
        HTTP_HEADER_ENABLE_CACHE_CONTROL("httpHeaderEnableCacheControl", "true",
            RegisteredServicePropertyGroups.HTTP_HEADERS, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether CAS should inject cache control headers into the response when this service is in process."),
        /**
         * Whether CAS should inject xcontent options headers into the response when this service is in process.
         */
        HTTP_HEADER_ENABLE_XCONTENT_OPTIONS("httpHeaderEnableXContentOptions", "true",
            RegisteredServicePropertyGroups.HTTP_HEADERS, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether CAS should inject xcontent options headers into the response when this service is in process."),
        /**
         * Whether CAS should inject strict transport security headers into the response when this service is in process.
         */
        HTTP_HEADER_ENABLE_STRICT_TRANSPORT_SECURITY("httpHeaderEnableStrictTransportSecurity", "true",
            RegisteredServicePropertyGroups.HTTP_HEADERS, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether CAS should inject strict transport security headers into the response when this service is in process."),
        /**
         * Control the header value CAS should use when injecting strict transport security headers into the response when this service is in process.
         */
        HTTP_HEADER_STRICT_TRANSPORT_SECURITY("httpHeaderStrictTransportSecurity", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.HTTP_HEADERS, RegisteredServicePropertyTypes.STRING,
            "Control the header value CAS should use when injecting strict transport security headers into the response when this service is in process."),
        /**
         * Whether CAS should inject xframe options headers into the response when this service is in process.
         */
        HTTP_HEADER_ENABLE_XFRAME_OPTIONS("httpHeaderEnableXFrameOptions", "true",
            RegisteredServicePropertyGroups.HTTP_HEADERS, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether CAS should inject xframe options headers into the response when this service is in process."),
        /**
         * Whether CAS should override xframe options headers into the response when this service is in process.
         */
        HTTP_HEADER_XFRAME_OPTIONS("httpHeaderXFrameOptions", "DENY",
            RegisteredServicePropertyGroups.HTTP_HEADERS, RegisteredServicePropertyTypes.STRING,
            "Whether CAS should override xframe options headers into the response when this service is in process."),
        /**
         * Whether CAS should inject content security policy headers into the response when this service is in process.
         */
        HTTP_HEADER_ENABLE_CONTENT_SECURITY_POLICY("httpHeaderEnableContentSecurityPolicy", "true",
            RegisteredServicePropertyGroups.HTTP_HEADERS, RegisteredServicePropertyTypes.STRING,
            "Whether CAS should inject content security policy headers into the response when this service is in process."),
        /**
         * Whether CAS should inject xss protection headers into the response when this service is in process.
         */
        HTTP_HEADER_ENABLE_XSS_PROTECTION("httpHeaderEnableXSSProtection", "true",
            RegisteredServicePropertyGroups.HTTP_HEADERS, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether CAS should inject xss protection headers into the response when this service is in process."),
        /**
         * Whether CAS should allow credentials in CORS requests.
         */
        CORS_ALLOW_CREDENTIALS("corsAllowCredentials", "false",
            RegisteredServicePropertyGroups.CORS, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether CAS should allow credentials in CORS requests."),
        /**
         * Define the max-age property for CORS requests.
         */
        CORS_MAX_AGE("corsMaxAge", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.CORS, RegisteredServicePropertyTypes.INTEGER,
            "Define the max-age property for CORS requests."),
        /**
         * Define allowed origins for CORS requests. Cannot use * when credentials are allowed.
         */
        CORS_ALLOWED_ORIGINS("corsAllowedOrigins", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.CORS, RegisteredServicePropertyTypes.STRING,
            "Define allowed origins for CORS requests. Cannot use * when credentials are allowed."),
        /**
         * Define patterns of allowed origins for CORS requests. (e.g.
         * 'https://*.example.com') Patterns can be used when credentials are allowed.
         */
        CORS_ALLOWED_ORIGIN_PATTERNS("corsAllowedOriginPatterns", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.CORS, RegisteredServicePropertyTypes.STRING,
            "Define patterns of allowed origins. (e.g.'https://*.example.com') Patterns can be used when credentials are allowed."),
        /**
         * Define allowed methods for CORS requests.
         */
        CORS_ALLOWED_METHODS("corsAllowedMethods", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.CORS, RegisteredServicePropertyTypes.STRING,
            "Define allowed methods for CORS requests. The special value * allows all methods."),
        /**
         * Define allowed headers for CORS requests.
         */
        CORS_ALLOWED_HEADERS("corsAllowedHeaders", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.CORS, RegisteredServicePropertyTypes.STRING,
            "Define exposed headers in the response for CORS requests. Set the list of headers that a pre-flight "
                + "request can list as allowed for use during an actual request. The special value "
                + "`*` allows actual requests to send any header."),
        /**
         * Define exposed headers in the response for CORS requests.
         */
        CORS_EXPOSED_HEADERS("corsExposedHeaders", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.CORS, RegisteredServicePropertyTypes.STRING,
            "List of response headers that a response might have and can be exposed. "
                + "The special value `*` allows all headers to be exposed for non-credentialed requests."),
        /**
         * Indicate binding type, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_FORCE_AUTHN("forceAuthn", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN, RegisteredServicePropertyTypes.BOOLEAN,
            "Indicate binding type, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate binding type, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_PASSIVE_AUTHN("passiveAuthn", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN, RegisteredServicePropertyTypes.BOOLEAN,
            "Indicate binding type, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate binding type, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_AUTHN_REQUEST_BINDING_TYPE("AuthnRequestBindingType", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.STRING,
            "Indicate binding type, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate assertion consumer service index, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_ASSERTION_CONSUMER_SERVICE_INDEX("AssertionConsumerServiceIndex", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.LONG,
            "Indicate assertion consumer service index, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate attribute consuming service index, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_ATTRIBUTE_CONSUMING_SERVICE_INDEX("AttributeConsumingServiceIndex", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.LONG,
            "Indicate attribute consuming service index, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate comparison type when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_COMPARISON_TYPE("ComparisonType", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.STRING,
            "Indicate comparison type when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate name id policy format, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_NAME_ID_POLICY_FORMAT("NameIdPolicyFormat", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.STRING,
            "Indicate name id policy format, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate name id policy allow create, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_NAME_ID_POLICY_ALLOW_CREATE("NameIdPolicyAllowCreate", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.BOOLEAN,
            "Indicate name id policy allow create, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate provider name, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_PROVIDER_NAME("ProviderName", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.STRING,
            "Indicate provider name, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate issuer format, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_ISSUER_FORMAT("IssuerFormat", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.STRING,
            "Indicate issuer format, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate whether name qualifier should be used, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_USE_NAME_QUALIFIER("UseNameQualifier", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.BOOLEAN,
            "Indicate whether name qualifier should be used, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate authn context class refs, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_AUTHN_CONTEXT_CLASS_REFS("AuthnContextClassRefs", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.SET,
            "Indicate authn context class refs, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate the name id attribute when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_NAME_ID_ATTRIBUTE("NameIdAttribute", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.STRING,
            "Indicate the name id attribute when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate whether assertions should be signed, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_WANTS_ASSERTIONS_SIGNED("WantsAssertionsSigned", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.BOOLEAN,
            "Indicate whether assertions should be signed, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate whether responses should be signed, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_WANTS_RESPONSES_SIGNED("WantsResponsesSigned", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.BOOLEAN,
            "Indicate whether responses should be signed, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate the maximum authentication lifetime to use, when using delegated authentication to saml2 identity providers.
         */
        DELEGATED_AUTHN_SAML2_MAXIMUM_AUTHN_LIFETIME("MaximumAuthenticationLifetime", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_SAML2, RegisteredServicePropertyTypes.LONG,
            "Indicate the maximum authentication lifetime to use, when using delegated authentication to saml2 identity providers."),
        /**
         * Indicate {@code max_age} to use, when using delegated authentication to OIDC OP.
         */
        DELEGATED_AUTHN_OIDC_MAX_AGE("max_age", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_OIDC, RegisteredServicePropertyTypes.INTEGER,
            "Indicate max_age to use, when using delegated authentication to OIDC OP"),
        /**
         * Indicate {@code scope} to use, when using delegated authentication to OIDC OP.
         */
        DELEGATED_AUTHN_OIDC_SCOPE("scope", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_OIDC, RegisteredServicePropertyTypes.STRING,
            "Indicate scope to use, when using delegated authentication to OIDC OP"),
        /**
         * Indicate {@code response_type} to use, when using delegated authentication to OIDC OP.
         */
        DELEGATED_AUTHN_OIDC_RESPONSE_TYPE("response_type", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_OIDC, RegisteredServicePropertyTypes.STRING,
            "Indicate response_type to use, when using delegated authentication to OIDC OP"),
        /**
         * Indicate {@code response_mode} to use, when using delegated authentication to OIDC OP.
         */
        DELEGATED_AUTHN_OIDC_RESPONSE_MODE("response_mode", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.DELEGATED_AUTHN_OIDC, RegisteredServicePropertyTypes.STRING,
            "Indicate response_mode to use, when using delegated authentication to OIDC OP"),
        /**
         * Whether captcha is enabled.
         */
        CAPTCHA_IP_ADDRESS_PATTERN("captchaIPAddressPattern", "true",
            RegisteredServicePropertyGroups.RECAPTCHA, RegisteredServicePropertyTypes.SET,
            "Whether reCAPTCHA should be activated when the remote-ip address matches any of the defined pattern(s)."),
        /**
         * Whether captcha is enabled.
         */
        CAPTCHA_ENABLED("captchaEnabled", "true",
            RegisteredServicePropertyGroups.RECAPTCHA, RegisteredServicePropertyTypes.BOOLEAN,
            "Whether reCAPTCHA should be enabled."),
        /**
         * Define SCIM oauth token.
         */
        SCIM_OAUTH_TOKEN("scimOAuthToken", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.SCIM, RegisteredServicePropertyTypes.STRING,
            "Define SCIM OAUTH token"),
        /**
         * Define SCIM username.
         */
        SCIM_USERNAME("scimUsername", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.SCIM, RegisteredServicePropertyTypes.STRING,
            "Define SCIM username"),
        /**
         * Define SCIM password.
         */
        SCIM_PASSWORD("scimPassword", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.SCIM, RegisteredServicePropertyTypes.STRING,
            "Define SCIM password"),
        /**
         * Define SCIM target.
         */
        SCIM_TARGET("scimTarget", StringUtils.EMPTY,
            RegisteredServicePropertyGroups.SCIM, RegisteredServicePropertyTypes.STRING,
            "Define SCIM target");

        private final String propertyName;

        private final String defaultValue;

        private final RegisteredServicePropertyGroups group;

        private final RegisteredServicePropertyTypes type;

        private final String description;

        /**
         * Does property belong to the requested group?
         *
         * @param group the group
         * @return true/false
         */
        @JsonIgnore
        public boolean isMemberOf(final RegisteredServicePropertyGroups group) {
            return this.group == group;
        }

        /**
         * Gets property value.
         *
         * @param service the service
         * @return the property value
         */
        @JsonIgnore
        public @Nullable RegisteredServiceProperty getPropertyValue(@Nullable final RegisteredService service) {
            if (isAssignedTo(service)) {
                val property = service.getProperties()
                    .entrySet()
                    .stream()
                    .filter(entry -> entry.getKey().equalsIgnoreCase(getPropertyName())
                        && StringUtils.isNotBlank(entry.getValue().value()))
                    .distinct()
                    .findFirst();
                if (property.isPresent()) {
                    return property.get().getValue();
                }
            }
            return null;
        }

        /**
         * Gets property value.
         *
         * @param <T>     the type parameter
         * @param service the service
         * @param clazz   the clazz
         * @return the property value
         */
        @JsonIgnore
        public @Nullable <T> T getPropertyValue(final RegisteredService service, final Class<T> clazz) {
            if (isAssignedTo(service)) {
                val prop = getPropertyValue(service);
                if (prop != null) {
                    return clazz.cast(prop.value());
                }
            }
            return null;
        }

        /**
         * Gets property values.
         *
         * @param <T>     the type parameter
         * @param service the service
         * @param clazz   the clazz
         * @return the property value
         */
        @JsonIgnore
        public @Nullable <T> T getPropertyValues(@Nullable final RegisteredService service, final Class<T> clazz) {
            if (isAssignedTo(service)) {
                val prop = getPropertyValue(service);
                if (prop != null) {
                    return clazz.cast(prop.getValues());
                }
            }
            return null;
        }

        /**
         * Gets property integer value.
         *
         * @param service the service
         * @return the property integer value
         */
        @JsonIgnore
        public int getPropertyIntegerValue(@Nullable final RegisteredService service) {
            if (isAssignedTo(service)) {
                val prop = getPropertyValue(service);
                if (prop != null) {
                    return Integer.parseInt(prop.value());
                }
            }
            return Integer.MIN_VALUE;
        }

        /**
         * Gets property long value.
         *
         * @param service the service
         * @return the property long value
         */
        @JsonIgnore
        public long getPropertyLongValue(@Nullable final RegisteredService service) {
            if (isAssignedTo(service)) {
                val prop = getPropertyValue(service);
                if (prop != null) {
                    return Long.parseLong(prop.value());
                }
            }
            return Long.MIN_VALUE;
        }

        /**
         * Gets property double value.
         *
         * @param service the service
         * @return the property double value
         */
        @JsonIgnore
        public double getPropertyDoubleValue(final RegisteredService service) {
            if (isAssignedTo(service)) {
                val prop = getPropertyValue(service);
                if (prop != null) {
                    return Double.parseDouble(prop.value());
                }
            }
            return Double.NaN;
        }

        /**
         * Gets property boolean value.
         *
         * @param service the service
         * @return the property boolean value
         */
        @JsonIgnore
        public boolean getPropertyBooleanValue(final RegisteredService service) {
            if (isAssignedTo(service)) {
                val prop = getPropertyValue(service);
                if (prop != null) {
                    return BooleanUtils.toBoolean(prop.value());
                }
            }
            return BooleanUtils.toBoolean(getDefaultValue());
        }

        /**
         * Check to see if the property is assigned to this service and is defined with a value.
         *
         * @param service registered service
         * @return true/false
         */
        @JsonIgnore
        public boolean isAssignedTo(@Nullable
                                    final RegisteredService service) {
            return isAssignedTo(service, s -> true);
        }

        /**
         * Is assigned to value.
         *
         * @param service     the service
         * @param valueFilter the filter
         * @return true/false
         */
        @JsonIgnore
        public boolean isAssignedTo(@Nullable final RegisteredService service, final Predicate<String> valueFilter) {
            return service != null && service.getProperties().entrySet()
                .stream()
                .anyMatch(entry -> entry.getKey().equalsIgnoreCase(getPropertyName())
                    && StringUtils.isNotBlank(entry.getValue().value())
                    && valueFilter.test(entry.getValue().value()));
        }

        /**
         * Gets typed property value.
         *
         * @param registeredService the registered service
         * @return the typed property value
         */
        public @Nullable Object getTypedPropertyValue(@Nullable final RegisteredService registeredService) {
            return switch (getType()) {
                case SET -> getPropertyValues(registeredService, Set.class);
                case INTEGER -> getPropertyIntegerValue(registeredService);
                case LONG -> getPropertyLongValue(registeredService);
                case BOOLEAN -> getPropertyBooleanValue(registeredService);
                default -> Objects.requireNonNull(getPropertyValue(registeredService)).value();
            };
        }

        /**
         * Is not assigned to boolean.
         *
         * @param service the service
         * @return true or false
         */
        @JsonIgnore
        public boolean isNotAssignedTo(final RegisteredService service) {
            return isNotAssignedTo(service, BooleanUtils::toBoolean);
        }

        /**
         * Is not assigned to boolean.
         *
         * @param service     the service
         * @param valueFilter the value filter
         * @return true or false
         */
        @JsonIgnore
        public boolean isNotAssignedTo(final RegisteredService service, final Predicate<String> valueFilter) {
            if (service == null || !service.getProperties().containsKey(getPropertyName())) {
                return true;
            }
            val property = service.getProperties().get(getPropertyName());
            return property == null || property.getValues().isEmpty() || property.getValues().stream().noneMatch(valueFilter);
        }
    }
}
