//-------------------------------------------------------------------------------------------------------------------------------------------------------------
//
// Copyright 2023 Apple Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//-------------------------------------------------------------------------------------------------------------------------------------------------------------

#ifndef RenderCore_hpp
#define RenderCore_hpp

#include <Metal/Metal.hpp>
#include <QuartzCore/QuartzCore.hpp>
#define IR_RUNTIME_METALCPP
#include <metal_irconverter_runtime/metal_irconverter_runtime.h>

#include "BumpAllocator.hpp"
#include "MeshUtils.hpp"
#include "ShaderPipelineBuilder.hpp"
#include <string>

constexpr size_t kMaxFramesInFlight = 3;

class RenderCore
{
public:
    RenderCore(MTL::Device* pDevice, const std::string& shaderSearchPath);
    ~RenderCore();

    // Set up the scene.
    void buildRenderPipelines(const std::string& shaderSearchPath);
    void buildComputePipelines(const std::string& shaderSearchPath);
    void buildTextures();
    void buildSamplers();
    void buildBuffers();

    // Call every frame to produce the animations.
    void raytrace(MTL::CommandBuffer* pCommandBuffer);
    void presentTexture(MTL::RenderCommandEncoder* pRenderEnc, MTL::Texture* pTexture);
    void updateWorld();
    void draw(MTL::RenderPassDescriptor* pRenderPass, CA::MetalDrawable* pDrawable);

    // Respond to configuration changes.
    void resizeDrawable(float width, float height);

private:
    MTL::Device* _pDevice;
    MTL::CommandQueue* _pCommandQueue;
    dispatch_semaphore_t _semaphore;

    // Pipeline states:
    MTL::ComputePipelineState* _pComputeRTPipeline;
    MTL::RenderPipelineState* _pPresentPipeline;

    // Assets:
    MTL::Texture*      _pTriangleTexture;
    MTL::SamplerState* _pSampler;
    IndexedMesh        _screenMesh;
    MTL::Heap*         _pScratch;
    
    // Data needed for ray tracing:
    NS::SharedPtr<MTL::AccelerationStructure> _primitiveAccelerationStructure;
    NS::SharedPtr<MTL::AccelerationStructure> _instanceAccelerationStructure;

    // Per-frame data:
    BumpAllocator* _bufferAllocator[kMaxFramesInFlight];

    // Animation data:
    int _frame;
    uint _animationIndex;

    // Render config:
    float _aspectRatio;
};

#endif /* RenderCore_hpp */
