# Adopting Function constants and Framebuffer fetch in HLSL

Convert HLSL shaders to Metal IR adopting function constants and framembuffer fetch

## Overview
This sample code project shows how to use Metal's function constants and framebuffer fetch features with HLSL shaders.

## Configure the sample code project
This project depends on **Metal Shader Converter**. Ensure you have version 3.0 or later installed. It searches for header files in `/usr/local/include` and relies on the **metal-shaderconverter** command line tool to convert DXIL to Metal IR. 

### Install DXC 
The sample project recompiles the HLSL files to DXIL when it finds the `dxc` command-line tool.

Follow these steps to clone and build `dxc` on your device:

```bash
git clone https://github.com/microsoft/DirectXShaderCompiler.git
cd DirectXShaderCompiler
git submodule update --init --recursive
mkdir build
cd build
cmake .. -G "Unix Makefiles" -DCMAKE_BUILD_TYPE=Release -C ../cmake/caches/PredefinedParams.cmake -DCMAKE_OSX_ARCHITECTURES="x86_64;arm64"
make -j8
```

After the build process completes, copy the files `bin/dxc` and `lib/libdxcompiler.dylib` to `/usr/local/bin/` and `/usr/local/lib` respectively, and create appropriate symlinks to each if required.

```bash
sudo ditto bin/dxc-3.7 /usr/local/bin/
sudo ditto lib/libdxcompiler.dylib /usr/local/lib/
sudo ln -s /usr/local/bin/dxc-3.7 /usr/local/bin/dxc
```

## Adopt Metal features in HLSL
### Compile HLSL to DXIL
This sample adopts metal features in the HLSL file named `triangles.hlsl` by including `Metal_HLSL.inc`. A copy of this file is installed by the **Metal Shader Converter 3.0** to `/usr/local/include/metal_irconverter_ext/Metal_HLSL.inc`.
```
#include "Metal_HLSL.inc"
``` 
To enable function constants and framebuffer fetch, `Metal_HLSL.inc` requires the definition of unique register spaces for each, using the macros `MTL_FUNCTION_CONSTANT_SPACE` and `MTL_FRAMEBUFFER_FETCH_SPACE` respectively. 
The sample does this during HLSL compilation using `dxc`'s command option `-D <macro>=<value>`
```bash
MTL_FRAMEBUFFER_FETCH_SPACE=2147420893
MTL_FUNCTION_CONSTANT_SPACE=2147420894

MACROS="-D MTL_FUNCTION_CONSTANT_SPACE=${MTL_FUNCTION_CONSTANT_SPACE} -D MTL_FRAMEBUFFER_FETCH_SPACE=${MTL_FRAMEBUFFER_FETCH_SPACE}"
```
Once these macros are defined, the features are availabe for use in HLSL. Function constants are declared with the macro `MTL_FUNCTION_CONSTANT(type, variable, index)` and the framebuffer fetch call is implemented with the macro `MTL_LOAD_FRAMEBUFFER(attachment_index, data_type)`. 
The sample uses both of these for its pixel shader.
```c
#include "triangles.h" // Defines function indices with prefix `FC_IDX_`

MTL_FUNCTION_CONSTANT(float4, ConstantColor, FC_IDX_FRAGMENT_CONSTANT_COLOR);
MTL_FUNCTION_CONSTANT(int, logicOpMode, FC_IDX_FRAGMENT_LOGIC_OP_MODE);
MTL_FUNCTION_CONSTANT(int, blendState, FC_IDX_FRAGMENT_BLEND_STATE);

half4 pixelShader() : SV_Target0
{
    half4 sourceColor = ConstantColor;
    if (blendState == LogicBlendStateDisabled)
    {
        return sourceColor;
    }

    // load pixel color at attachment 0
    half4 destColor = MTL_LOAD_FRAMEBUFFER(0, half4);
    float4 maxValue = float4(0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF);

    uint4 sourceColorUint = uint4(float4(sourceColor) * maxValue);
    uint4 destColorUint = uint4(float4(destColor) * maxValue);

    // Using logic operations, blend source color from function constant with color at attachment
    LOGIC_OP op = (LOGIC_OP)logicOpMode;
    uint4 finalColor = logicOp(op, sourceColorUint, destColorUint);
    half4 pixelColor = half4(float4(finalColor) / maxValue);
    return pixelColor;
}
```
##### IMPORTANT 
These features leverage the D3D12 resource binding model. Hence, HLSL shaders that define root signatures need to explicitly include these unique registers spaces as needed.
The sample includes the toggle macro `#define USE_ROOT_SIGNATURE 0` which demonstrates how to do this when enabled.

### Convert DXIL to MetalLib
Once the HLSL shader is compiled to DXIL with metal features enabled, the sample then passes these unique register space values to `metal-shaderconverter` using the command options:
```bash
--framebuffer-fetch-register-space=${MTL_FRAMEBUFFER_FETCH_SPACE} 
--function-constant-register-space=${MTL_FUNCTION_CONSTANT_SPACE}
```

## Set function constant values at runtime
Finally, the sample ensures function constant values are initialized when building pipeline variants, using the function `IRRuntimeSetFunctionConstantValue(MTLFunctionConstantValues*, index, value)` 

```c
#import "../HLSL/triangles.h" // Defines function indices with prefix `FC_IDX_`

LOGIC_OP logicOp = (LOGIC_OP)op;
IRRuntimeFunctionConstantValue logicOpFC = { .i0 = logicOp };
IRRuntimeSetFunctionConstantValue(functionConstantValues, FC_IDX_FRAGMENT_LOGIC_OP_MODE, &logicOpFC);
...
IRRuntimeFunctionConstantValue blendStateFC = { .i0 = blendState };
IRRuntimeSetFunctionConstantValue(functionConstantValues, FC_IDX_FRAGMENT_BLEND_STATE, &blendStateFC);
...
IRRuntimeFunctionConstantValue colorFC = { .f0 = color->x, .f1 = color->y, .f2 = color->z, .f3 = color->w };
IRRuntimeSetFunctionConstantValue(functionConstantValues, FC_IDX_FRAGMENT_CONSTANT_COLOR, &colorFC);
```
